# AEER-UA Backend API

This is the backend API for the AEER-UA Event Management system. It is built with Node.js, Express, and PostgreSQL.

## Features

- **RESTful API**: Full CRUD operations for events.
- **Authentication**: JWT-based authentication.
- **RBAC**: Role-Based Access Control (Admin, Participant, Member, Company).
- **Database**: PostgreSQL with optimized schema.
- **Security**: Helmet for headers, bcrypt for password hashing.

## Setup

1.  **Install Dependencies**:
    ```bash
    npm install
    ```

2.  **Environment Variables**:
    Create a `.env` file in the root of the `backend` folder (copy from `.env.example` if available) and set the following:
    ```env
    PORT=5000
    DATABASE_URL=postgresql://user:password@localhost:5432/aeer_events
    JWT_SECRET=your_secure_jwt_secret
    NODE_ENV=development
    ```

3.  **Database Initialization**:
    Make sure your PostgreSQL server is running and the database `aeer_events` exists (or whatever name you chose).
    Run the initialization script to create tables and default roles:
    ```bash
    npm run init-db
    ```

4.  **Run Server**:
    ```bash
    npm run dev
    ```

## API Endpoints

### Auth
- `POST /api/auth/register`: Register a new user.
- `POST /api/auth/login`: Login and get JWT token.

### Events
- `GET /api/events`: Get all events.
- `GET /api/events/:id`: Get event by ID.
- `POST /api/events`: Create event (Admin only).
- `PUT /api/events/:id`: Update event (Admin only).
- `DELETE /api/events/:id`: Delete event (Admin only).

## Integration Guidelines for Frontend

1.  **Base URL**: Set your API base URL to `http://localhost:5000/api` (or your production URL).
2.  **Authentication**:
    - On login, store the `token` from the response in `localStorage` or `HttpOnly` cookie.
    - Include the token in the `Authorization` header for protected requests:
      ```javascript
      headers: {
        'Authorization': `Bearer ${token}`
      }
      ```
3.  **Error Handling**:
    - The API returns errors in the format `{ message: "Error description" }`.
    - Handle 401 (Unauthorized) by redirecting to login.
    - Handle 403 (Forbidden) by showing a "Permission Denied" message.

## Deployment Considerations

- **Environment Variables**: Ensure all secrets (DB credentials, JWT secret) are set in the production environment variables, not hardcoded.
- **Database**: Use a managed PostgreSQL service (e.g., Supabase, AWS RDS, Heroku Postgres) for reliability and backups.
- **Security**:
    - Enable HTTPS (SSL/TLS).
    - Set `NODE_ENV=production`.
    - Use a reverse proxy (Nginx) if deploying on a VPS.
- **Process Management**: Use `pm2` to keep the application running.
    ```bash
    npm install -g pm2
    pm2 start app.js --name "aeer-backend"
    ```
