const db = require('../config/db');
const { sendSubmissionNotification } = require('../services/emailService');

const getAllEvents = async (req, res, next) => {
  try {
    const result = await db.query('SELECT * FROM events ORDER BY date ASC');
    res.json(result.rows);
  } catch (error) {
    next(error);
  }
};

const getEventById = async (req, res, next) => {
  const { id } = req.params;
  try {
    const result = await db.query('SELECT * FROM events WHERE id = $1', [id]);
    if (result.rows.length === 0) {
      return res.status(404).json({ message: 'Event not found' });
    }
    res.json(result.rows[0]);
  } catch (error) {
    next(error);
  }
};

const createEvent = async (req, res, next) => {
  const { title, description, date, location, image_url } = req.body;
  const createdBy = req.user.id; // From JWT

  try {
    const result = await db.query(
      'INSERT INTO events (title, description, date, location, image_url, created_by) VALUES ($1, $2, $3, $4, $5, $6) RETURNING *',
      [title, description, date, location, image_url, createdBy]
    );
    res.status(201).json(result.rows[0]);
  } catch (error) {
    next(error);
  }
};

const updateEvent = async (req, res, next) => {
  const { id } = req.params;
  const { title, description, date, location, image_url } = req.body;

  try {
    const result = await db.query(
      'UPDATE events SET title = $1, description = $2, date = $3, location = $4, image_url = $5, updated_at = CURRENT_TIMESTAMP WHERE id = $6 RETURNING *',
      [title, description, date, location, image_url, id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ message: 'Event not found' });
    }
    res.json(result.rows[0]);
  } catch (error) {
    next(error);
  }
};

const deleteEvent = async (req, res, next) => {
  const { id } = req.params;
  try {
    const result = await db.query('DELETE FROM events WHERE id = $1 RETURNING *', [id]);
    if (result.rows.length === 0) {
      return res.status(404).json({ message: 'Event not found' });
    }
    res.json({ message: 'Event deleted successfully' });
  } catch (error) {
    next(error);
  }
};

const registerForEvent = async (req, res, next) => {
  const { id } = req.params;
  const userId = req.user ? req.user.id : null;

  try {
    // Check if event exists and is active
    const eventResult = await db.query('SELECT * FROM events WHERE id = $1', [id]);
    if (eventResult.rows.length === 0) {
      return res.status(404).json({ message: 'Event not found' });
    }
    const event = eventResult.rows[0];

    if (!event.is_active) {
      return res.status(400).json({ message: 'Registration for this event is closed.' });
    }

    // Check if already registered (only for logged in users)
    if (userId) {
      const checkReg = await db.query('SELECT * FROM registrations WHERE user_id = $1 AND event_id = $2', [userId, id]);
      if (checkReg.rows.length > 0) {
        return res.status(400).json({ message: 'Already registered for this event' });
      }
    }

    // Register with details
    const details = req.body; // Capture form data
    await db.query('INSERT INTO registrations (user_id, event_id, details) VALUES ($1, $2, $3)', [userId, id, details]);

    // Get user details for notification
    let user = { username: 'Guest', email: details.email };
    if (userId) {
      const userResult = await db.query('SELECT * FROM users WHERE id = $1', [userId]);
      user = userResult.rows[0];
    }

    // Send Email Notification
    await sendSubmissionNotification(event.title, {
      name: details.contactName || details.fullName || user.username, // Prefer form data
      email: details.email || user.email,
      phone: details.phone,
      organization: details.orgName || details.organization,
      ...details
    });

    res.status(201).json({ message: 'Registration successful' });
  } catch (error) {
    next(error);
  }
};

const toggleEventStatus = async (req, res, next) => {
  const { id } = req.params;
  const { is_active } = req.body; // Expect boolean

  try {
    const result = await db.query(
      'UPDATE events SET is_active = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2 RETURNING *',
      [is_active, id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ message: 'Event not found' });
    }
    
    res.json(result.rows[0]);
  } catch (error) {
    next(error);
  }
};

const getEventRegistrations = async (req, res, next) => {
  const { id } = req.params;
  try {
    // Updated query to handle guest users (LEFT JOIN) and extract details
    const result = await db.query(`
      SELECT 
        r.id,
        COALESCE(u.username, r.details->>'fullName', r.details->>'contactName', 'Guest') as username,
        COALESCE(u.email, r.details->>'email') as email,
        r.registered_at,
        r.status,
        r.details
      FROM registrations r
      LEFT JOIN users u ON r.user_id = u.id
      WHERE r.event_id = $1
      ORDER BY r.registered_at DESC
    `, [id]);
    
    res.json(result.rows);
  } catch (error) {
    next(error);
  }
};



const getAllRegistrations = async (req, res, next) => {
  const { type, eventId, search, page = 1, limit = 50 } = req.query;
  const offset = (page - 1) * limit;
  
  let query = `
    SELECT 
      r.id, 
      r.event_id, 
      e.title as event_title, 
      r.user_id, 
      r.registered_at, 
      r.status, 
      r.details,
      COALESCE(u.username, r.details->>'fullName', r.details->>'contactName', 'Guest') as username,
      COALESCE(u.email, r.details->>'email') as email
    FROM registrations r
    JOIN events e ON r.event_id = e.id
    LEFT JOIN users u ON r.user_id = u.id
    WHERE 1=1
  `;
  const params = [];
  let paramCount = 1;

  if (type) {
    query += ` AND r.details->>'formType' = $${paramCount}`;
    params.push(type);
    paramCount++;
  }

  if (eventId) {
    query += ` AND r.event_id = $${paramCount}`;
    params.push(eventId);
    paramCount++;
  }

  if (search) {
    query += ` AND (
      r.details->>'email' ILIKE $${paramCount} OR
      r.details->>'fullName' ILIKE $${paramCount} OR
      r.details->>'contactName' ILIKE $${paramCount} OR
      r.details->>'orgName' ILIKE $${paramCount} OR
      u.username ILIKE $${paramCount} OR
      u.email ILIKE $${paramCount}
    )`;
    params.push(`%${search}%`);
    paramCount++;
  }

  query += ` ORDER BY r.registered_at DESC LIMIT $${paramCount} OFFSET $${paramCount + 1}`;
  params.push(limit, offset);

  try {
    const result = await db.query(query, params);
    res.json(result.rows);
  } catch (error) {
    next(error);
  }
};

const updateRegistration = async (req, res, next) => {
  const { id } = req.params;
  const { status, details } = req.body;

  try {
    let query = 'UPDATE registrations SET ';
    const params = [];
    let paramCount = 1;
    const updates = [];

    if (status) {
      updates.push(`status = $${paramCount}`);
      params.push(status);
      paramCount++;
    }

    if (details) {
      updates.push(`details = $${paramCount}`);
      params.push(details);
      paramCount++;
    }

    if (updates.length === 0) {
      return res.status(400).json({ message: 'No fields to update' });
    }

    query += updates.join(', ');
    query += ` WHERE id = $${paramCount} RETURNING *`;
    params.push(id);

    const result = await db.query(query, params);

    if (result.rows.length === 0) {
      return res.status(404).json({ message: 'Registration not found' });
    }

    res.json(result.rows[0]);
  } catch (error) {
    next(error);
  }
};

const deleteRegistration = async (req, res, next) => {
  const { id } = req.params;
  try {
    const result = await db.query('DELETE FROM registrations WHERE id = $1 RETURNING *', [id]);
    if (result.rows.length === 0) {
      return res.status(404).json({ message: 'Registration not found' });
    }
    res.json({ message: 'Registration deleted successfully' });
  } catch (error) {
    next(error);
  }
};

module.exports = { 
  getAllEvents, 
  getEventById, 
  createEvent, 
  updateEvent, 
  deleteEvent,
  registerForEvent,
  toggleEventStatus,
  getEventRegistrations,
  getAllRegistrations,
  updateRegistration,
  deleteRegistration
};
